#pragma once

/// @file
/// @brief Define encryption API functions
///
/// @copyright Copyright 2023 InfoTeCS.

#ifdef EncryptFile
#    undef EncryptFile
#endif

#ifdef DecryptFile
#    undef DecryptFile
#endif

#include <stdint.h>
#include <stddef.h>

#if ( __GNUC__ >= 4 ) || defined( __clang__ )
#    define EncryptionApiExport __attribute__( ( visibility( "default" ) ) )
#else
#    define EncryptionApiExport
#endif

#ifndef EXPORT_VOID
#    ifdef __cplusplus
#        define EXPORT_VOID
#    else
#        define EXPORT_VOID void
#    endif
#endif

/// @brief Command execution result
typedef struct
{
    /// @brief Function output code (0 - success)
    uint32_t code;

    /// @brief Error message (blank if success)
    /// @remarks Does not require the memory release. The message is stored till the next command call.
    const char* message;
} EncryptionApiReturnCode;

/// @brief Raw data details
typedef struct
{
    /// @brief Data array
    /// @remarks Requires the memory release
    const uint8_t* bytes;

    /// @brief Size of the array
    size_t size;
} VpnRawData;

/*! \defgroup api_ver1 Encryption API functions for version 1 and higher
    @{
*/
/// @brief Encrypt file using the host keys
/// @param[in] nodeId - ID of the host the encryption is performed for
/// @param[in] filePath - path to the file to be encrypted
/// @param[out] encryptionData - encryption data used to encrypt the file
/// @remarks The encryption data memory is allocated dynamically and must be released using ReleaseRawData( encryptionData )
/// @return Command execution result
typedef EncryptionApiReturnCode ( *EncryptFile )( uint32_t nodeId, const char* filePath, VpnRawData** encryptionData );

/// @brief Decrypt file using the host keys
/// @param[in] nodeId - ID of the host the encryption is performed for
/// @param[in] filePath - path to the file to be decrypted
/// @param[in] encryptionData - encryption data used to decrypt the file
/// @return Command execution result
typedef EncryptionApiReturnCode ( *DecryptFile )( uint32_t nodeId,
                                                  const char* filePath,
                                                  const VpnRawData* encryptionData );

/// @brief Encrypt blob using the host keys
/// @param[in] nodeId - ID of the host the encryption is performed for
/// @param[in] sourceBlob - blob to encrypt
/// @param[out] destinationBlob - encrypted blob
/// @param[out] encryptionData - encryption data used to encrypt the blob
/// @remarks The encrypted blob memory is allocated dynamically and must be released using ReleaseRawData( destinationBlob )
/// @remarks The encryption data memory is allocated dynamically and must be released using ReleaseRawData( encryptionData )
/// @return Command execution result
typedef EncryptionApiReturnCode ( *EncryptBlob )( uint32_t nodeId,
                                                  const VpnRawData* sourceBlob,
                                                  VpnRawData** destinationBlob,
                                                  VpnRawData** encryptionData );

/// @brief Decrypt blob using the host keys
/// @param[in] nodeId - ID of the host the encryption is performed for
/// @param[in] sourceBlob - blob to decrypt
/// @param[out] destinationBlob - decrypted blob
/// @param[in] encryptionData - encryption data used to decrypt the blob
/// @remarks The decrypted blob memory is allocated dynamically and must be released using ReleaseRawData( destinationBlob )
/// @return Command execution result
typedef EncryptionApiReturnCode ( *DecryptBlob )( uint32_t nodeId,
                                                  const VpnRawData* sourceBlob,
                                                  VpnRawData** destinationBlob,
                                                  const VpnRawData* encryptionData );

/// @brief Release memory allocated to the raw data
/// @param[in] data - data to be released
/// @return Command execution result
typedef EncryptionApiReturnCode ( *ReleaseRawData )( VpnRawData* data );
/*! @} */

/// @brief Encryption API functions (described above)
typedef struct
{
    // API_VERSION_1
    EncryptFile encryptFile;
    DecryptFile decryptFile;
    EncryptBlob encryptBlob;
    DecryptBlob decryptBlob;
    ReleaseRawData releaseRawData;
} ItcsEncryptionApi;

#ifdef __cplusplus
extern "C" {
#endif

/// @brief Get the current encryption API version
/// @remarks When updating API, check it for compatibility; otherwise, correct operation is not guaranteed
/// @return Current encryption API version
EncryptionApiExport uint32_t GetEncryptionApiVersion( EXPORT_VOID );

/// @brief Get the encryption API structure pointer
/// @return Pointer to the encryption API structure
EncryptionApiExport ItcsEncryptionApi* GetEncryptionApi( EXPORT_VOID );

#ifdef __cplusplus
}
#endif
