#pragma once

/// @file
/// @brief Define API functions for the Business Mail envelopes
///
/// @copyright Copyright 2021 InfoTeCS.

#include <stdint.h>
#include <stddef.h>
#include <time.h>

#if ( __GNUC__ >= 4 ) || defined( __clang__ )
#   define BmApiExport __attribute__( ( visibility( "default" ) ) )
#else
#   define BmApiExport
#endif

#ifndef EXPORT_VOID
#    ifdef __cplusplus
#        define EXPORT_VOID
#    else
#        define EXPORT_VOID void
#    endif
#endif

enum BusinessMailApiVersion
{
    BUSINESS_MAIL_API_VERSION = 2
};

/// @brief Envelope type
enum VpnBusinessMailEnvelopeType
{
    BmLetter = 0, ///> Message
    BmReceipt = 1 ///> Receipt
};

/// @brief Message recipient type
enum VpnBusinessMailRecipientType
{
    BmRecipientTo = 0,    ///> The "To" recipient
    BmRecipientCopy = 1,  ///> The "Copy" recipient
    BmRecipientHidden = 2 ///> The "CC" recipient
};

/// @brief Letter attachment flags
enum VpnBusinessMailAttachmentFlags
{
    BmAttachmentNoFlags = 0,      ///< Attachment has no special flags
    BmAttachmentSignature = 0x08, ///< Attachment is a PKCS#7 signature
    BmAttachmentSigned = 0x20,    ///< Attachment is signed with a detached PKCS#7 signature
};

/// @brief Command execution result
typedef struct
{
    /// @brief Function output code (0 - success)
    uint32_t code;

    /// @brief Error message (blank if success)
    /// @remarks Does not require the memory release. The message is stored till the next command call.
    const char* message;
} BusinessMailApiReturnCode;

///@brief Business Mail message details
typedef struct
{
    uint32_t sender;        ///> Sender host ID
    uint32_t* recipientIds; ///> IDs of all the recipient hosts (for an incoming message, the current recipient is the first in the list)
    enum VpnBusinessMailRecipientType* recipientTypes; ///> Recipient type
    size_t recipientsCount;                            ///> Recipient count
    const char* regNumber;                             ///> Message registration number
    const char* subject;                               ///> Message subject
    const char* message;                               ///> Message
    time_t creationTime;                               ///> Creation date
    time_t sendTime;                                   ///> Send date
    const char** attachmentPaths;                      ///> Attachment paths
    const char** attachmentNames;                      ///> Attachment names
    size_t attachmentsCount;                           ///> Atachment count
} VpnBusinessMailLetterData;

///@brief Business Mail receipt details
typedef struct
{
    uint32_t sender;        ///> Sender host ID
    uint32_t recipient;     ///> Recipient host ID
    uint8_t type;           ///> Receipt type
    const char* letterName; ///> Message envelope name
    time_t creationTime;    ///> Creation date
} VpnBusinessMailReceiptData;

///@brief Business Mail message details version 2
typedef struct
{
    uint32_t sender;        ///< Sender host ID
    uint32_t* recipientIds; ///< IDs of all the recipient hosts (for an incoming message, the current recipient is the first in the list)
    enum VpnBusinessMailRecipientType* recipientTypes; ///< Recipient type
    uint32_t recipientsCount;                          ///< Recipient count
    const char* regNumber;                             ///< Message registration number
    const char* subject;                               ///< Message subject
    const char* message;                               ///< Message
    uint32_t creationTime;                             ///< Creation date
    uint32_t sendTime;                                 ///< Sending date
    const char** attachmentPaths;                      ///< Attachment paths
    const char** attachmentNames;                      ///< Attachment names
    uint32_t attachmentsCount;                         ///< Attachment count
    uint32_t* attachmentFlags;                         ///< Attachment flags
} VpnBusinessMailLetterDataV2;

/*! \defgroup api_ver1 BM API functions for version 1 and higher
    @{
*/
/// @brief Get the Business Mail envelope list
/// @param[in] type - envelope type
/// @param[out] envelopes - envelope list
/// @param[out] size - envelope count
/// @remarks The envelopes memory is allocated dynamically and must be released manually
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *GetBusinessMailEnvelopes )( enum VpnBusinessMailEnvelopeType type,
                                                                 char*** envelopes,
                                                                 size_t* size );

/// @brief Delete the Business Mail envelope
/// @param[in] envelope - envelope name
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *RemoveBusinessMailEnvelope )( const char* envelope );

/// @brief Unpack the Business Mail message
/// @param[in] envelope - envelope name
/// @param[in] attachmentsDir - directory to copy the attachments, if any
/// @param[out] bmData - message envelope contents
/// @remarks The bmData memory is allocated dynamically and must be released using ReleaseVpnBusinessMailLetterData( bmData )
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *UnpackBusinessMailLetter )( const char* envelope,
                                                                 const char* attachmentsDir,
                                                                 VpnBusinessMailLetterData** bmData );

/// @brief Unpack the Business Mail receipt
/// @param[in] envelope - receipt envelope name
/// @param[out] bmData - receipt contents
/// @remarks The bmData memory is allocated dynamically and must be released using ReleaseVpnBusinessMailReceiptData( bmData )
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *UnpackBusinessMailReceipt )( const char* envelope,
                                                                  VpnBusinessMailReceiptData** bmData );

/// @brief Create and send the Business Mail message
/// @param[in] bmData - message envelope contents
/// @param[out] envelopes - envelope names (sorted in order of recipients)
/// @param[out] size - envelope count
/// @remarks The envelopes memory is allocated dynamically and must be released manually
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *PackBusinessMailLetter )( VpnBusinessMailLetterData* bmData,
                                                               char*** envelopes,
                                                               size_t* size );

/// @brief Generate and send the Business Mail receipt
/// @param[in] bmData - receipt envelope contents
/// @param[out] envelope - envelope name
/// @remarks The envelope memory is allocated dynamically and must be released manually
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *PackBusinessMailReceipt )( VpnBusinessMailReceiptData* bmData, char** envelope );

/// @brief Release memory allocated to the message contents
/// @param[in] bmData - message envelope contents
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *ReleaseVpnBusinessMailLetterData )( VpnBusinessMailLetterData* bmData );

/// @brief Release memory allocated to the receipt contents
/// @param[in] bmData - receipt envelope contents
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *ReleaseVpnBusinessMailReceiptData )( VpnBusinessMailReceiptData* bmData );
/*! @} */ // api_ver1

/*! \defgroup api_ver2 BM API functions for version 2 and higher
    @{
*/
/// @brief Unpack the Business Mail message version 2
/// @param[in] envelope - envelope name
/// @param[in] attachmentsDir - directory to copy the attachments, if any
/// @param[out] bmData - message envelope contents
/// @remarks The bmData memory is allocated dynamically and must be released using ReleaseVpnBusinessMailLetterDataV2( bmData )
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *UnpackBusinessMailLetterV2 )( const char* envelope,
                                                                   const char* attachmentsDir,
                                                                   VpnBusinessMailLetterDataV2** bmData );

/// @brief Create and send the Business Mail message version 2
/// @param[in] bmData - message envelope contents
/// @param[out] envelopes - envelope names (sorted in order of recipients)
/// @param[out] size - envelope count
/// @remarks The envelopes memory is allocated dynamically and must be released manually
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *PackBusinessMailLetterV2 )( VpnBusinessMailLetterDataV2* bmData,
                                                                 char*** envelopes,
                                                                 uint32_t* size );

/// @brief Release memory allocated to the message version 2 contents
/// @param[in] bmData - message envelope contents
/// @return Command execution result
typedef BusinessMailApiReturnCode ( *ReleaseVpnBusinessMailLetterDataV2 )( VpnBusinessMailLetterDataV2* bmData );
/*! @} */ // api_ver2

/// @brief API functions for Business Mail envelopes (described above)
typedef struct
{
    // API_VERSION_1
    GetBusinessMailEnvelopes getBusinessMailEnvelopes;
    RemoveBusinessMailEnvelope removeBusinessMailEnvelope;
    UnpackBusinessMailLetter unpackBusinessMailLetter;
    UnpackBusinessMailReceipt unpackBusinessMailReceipt;
    PackBusinessMailLetter packBusinessMailLetter;
    PackBusinessMailReceipt packBusinessMailReceipt;
    ReleaseVpnBusinessMailLetterData releaseVpnBusinessMailLetterData;
    ReleaseVpnBusinessMailReceiptData releaseVpnBusinessMailReceiptData;
    // API_VERSION_2
    UnpackBusinessMailLetterV2 unpackBusinessMailLetterV2;
    PackBusinessMailLetterV2 packBusinessMailLetterV2;
    ReleaseVpnBusinessMailLetterDataV2 releaseVpnBusinessMailLetterDataV2;
} ItcsBusinessMailApi;

#ifdef __cplusplus
extern "C" {
#endif

/// @brief Get the current version of API for Business Mail envelopes
/// @remarks When updating API, check it for compatibility; otherwise, correct operation is not guaranteed
/// @return Current version of API for Business Mail envelopes
BmApiExport uint32_t GetBusinessMailApiVersion( EXPORT_VOID );

/// @brief Get the pointer to the structure of API for Business Mail envelopes
/// @return Pointer to the structure of API for Business Mail envelopes
BmApiExport ItcsBusinessMailApi* GetBusinessMailApi( EXPORT_VOID );

#ifdef __cplusplus
}
#endif
